#include <iostream>
#include <thread>
#include <string>
#include "er4commlib.h"
#include "er4commlib_errorcodes.h"

using namespace er4CommLib;

int main()
{

    // Detect the available devices
    std::vector<std::string> deviceIds;
    ErrorCodes_t detectError = detectDevices(deviceIds);

    if(detectError != Success){
        if(detectError == ErrorNoDeviceFound)
            std::cout << "Error device not found: " << std::hex << detectError << std::endl;
        else{
            std::cout << "Another error: " << std::hex << detectError << std::endl;
        }
        return -1;
    }

    for(int i = 0; i < (int)deviceIds.size(); i++){
        std::cout << "Detected device " + std::to_string(i)+ ": " + deviceIds[i] << std::endl;
    }

    // Connect to the available device
    connect(deviceIds);

    // Check the available voltage and current channels
    uint32_t voltageChannelsNum;
    uint32_t currentChannelsNum;
    getChannelsNumber(voltageChannelsNum, currentChannelsNum);
    std::cout << "Available voltage channels: " + std::to_string(voltageChannelsNum) << std::endl;
    std::cout << "Available current channels: " + std::to_string(currentChannelsNum) << std::endl;
    std::cout << std::endl;

    // Check the available current ranges
    std::vector<RangedMeasurement_t> currentRanges;
    std::vector<uint16_t> defaultOptions;
    getCurrentRanges(currentRanges, defaultOptions);
    std::cout << "Available current ranges "  << std::endl;
    for(int i = 0; i < (int)currentRanges.size(); i++){
        std::cout << "\t Current range " + std::to_string(i) + ": " + currentRanges[i].niceLabel() << std::endl;
    }
    std::cout << std::endl;

    // Check the available sampling rates
    std::vector<Measurement_t> samplingRates;
    uint16_t defaultOption;
    getSamplingRates(samplingRates, defaultOption);
    std::cout << "Available sampling rates "  << std::endl;
    for(int i = 0; i < (int)samplingRates.size(); i++){
        std::cout << "\t Sampling rate " + std::to_string(i) + ": " + samplingRates[i].niceLabel() << std::endl;
    }
    std::cout << std::endl;

    // set current range 2nA
    setCurrentRange(1);

    // set sampling rate 1.25kHz
    setSamplingRate(0);

    // turn on digital offset compensation on all channels
    std::cout << "Start digital offset compensation" << std::endl;
    digitalOffsetCompensation(currentChannelsNum, true);

    // wait for 5s
    std::this_thread::sleep_for(std::chrono::milliseconds(5000));

    // turn off digital offset compensation on all channels
    digitalOffsetCompensation(currentChannelsNum, false);
    std::cout << "Stop digital offset compensation" << std::endl;
    std::cout << std::endl;


    // Get the list of the available protocols
    std::vector<std::string> names;
    std::vector<std::string> images;
    std::vector<std::vector<uint16_t>> voltages;
    std::vector<std::vector<uint16_t>> times;
    std::vector<std::vector<uint16_t>> slopes;
    std::vector<std::vector<uint16_t>> frequencies;
    std::vector<std::vector<uint16_t>> adimensionals;
    getProtocolList(names, images, voltages, times, slopes, frequencies, adimensionals);
    std::cout << "Available protocols "  << std::endl;
    for(int i = 0; i < (int)names.size(); i++){
        std::cout << "\t Protocol " + std::to_string(i) + ": " + names[i] << std::endl;
    }
    std::cout << std::endl;


    // Choose the conductance protocol (ID = 3)
    int protocolId = 3;

    // Get the vectors containing the indexes of the conductance protocol features
    std::vector<uint16_t> conductProtVoltages = voltages[protocolId];
    std::vector<uint16_t> conductProtTimes = times[protocolId];
    std::vector<uint16_t> conductProtSlopes = slopes[protocolId];
    std::vector<uint16_t> conductProtFrequencies = frequencies[protocolId];
    std::vector<uint16_t> conductProtAdimensionals = adimensionals[protocolId];

    // feature conductance protocol
    std::cout << "Indexes available for the conductance protocol features"  << std::endl;
    if(conductProtVoltages.size() == 0){
        std::cout << "\t No voltages available for this protocol "  << std::endl;
    } else {
        for(int i = 0; i < (int)conductProtVoltages.size(); i++){
            std::cout << "\t Voltages " + std::to_string(i) + ": " + std::to_string(conductProtVoltages[i]) << std::endl;
        }
    }

    if(conductProtTimes.size() == 0){
        std::cout << "\t No times available for this protocol "  << std::endl;
    } else {
        for(int i = 0; i < (int)conductProtTimes.size(); i++){
            std::cout << "\t Times " + std::to_string(i) + ": " + std::to_string(conductProtTimes[i]) << std::endl;
        }
    }

    if(conductProtSlopes.size() == 0){
        std::cout << "\t No slopes available for this protocol "  << std::endl;
    } else {
        for(int i = 0; i < (int)conductProtSlopes.size(); i++){
            std::cout << "\t Slopes " + std::to_string(i) + ": " + std::to_string(conductProtSlopes[i]) << std::endl;
        }
    }

    if(conductProtFrequencies.size() == 0){
        std::cout << "\t No frequencies available for this protocol "  << std::endl;
    } else {
        for(int i = 0; i < (int)conductProtFrequencies.size(); i++){
            std::cout << "\t Frequencies " + std::to_string(i) + ": " + std::to_string(conductProtFrequencies[i]) << std::endl;
        }
    }

    if(conductProtAdimensionals.size() == 0){
        std::cout << "\t No adimensionals available for this protocol "  << std::endl;
    } else {
        for(int i = 0; i < (int)conductProtAdimensionals.size(); i++){
            std::cout << "\t Adimensionals " + std::to_string(i) + ": " + std::to_string(conductProtAdimensionals[i]) << std::endl;
        }
    }
    std::cout << std::endl;


    std::cout << "Available conductance protocol features to select with the previous indexes"  << std::endl;
    std::vector<std::string> voltageNames;
    std::vector<RangedMeasurement_t> voltageRanges;
    std::vector<Measurement_t> defaultValues;
    getProtocolVoltage(voltageNames, voltageRanges, defaultValues);
    for(int i = 0; i < (int)conductProtVoltages.size(); i++){
        std::cout << "\t Voltage name " + std::to_string(i) + ": " + voltageNames[conductProtVoltages[i]] << std::endl;
        std::cout << "\t\t Range: " + voltageRanges[conductProtVoltages[i]].niceLabel() << std::endl;
        std::cout << "\t\t min: " + voltageRanges[conductProtVoltages[i]].getMin().niceLabel() << std::endl;
        std::cout << "\t\t max: " + voltageRanges[conductProtVoltages[i]].getMax().niceLabel() << std::endl;
    }

    std::vector<std::string> timeNames;
    std::vector<RangedMeasurement_t> timeRanges;
    getProtocolTime(timeNames, timeRanges, defaultValues);
    for(int i = 0; i < (int)conductProtTimes.size(); i++){
        std::cout << "\t Time name " + std::to_string(i) + ": " + timeNames[conductProtTimes[i]] << std::endl;
        std::cout << "\t\t Range: " + timeRanges[conductProtTimes[i]].niceLabel() << std::endl;
        std::cout << "\t\t min: " + timeRanges[conductProtTimes[i]].getMin().niceLabel() << std::endl;
        std::cout << "\t\t max: " + timeRanges[conductProtTimes[i]].getMax().niceLabel() << std::endl;
    }


    std::vector<std::string> adimensionalNames;
    std::vector<RangedMeasurement_t> adimensionalRanges;
    getProtocolAdimensional(adimensionalNames, adimensionalRanges, defaultValues);
    for(int i = 0; i < (int)conductProtAdimensionals.size(); i++){
        std::cout << "\t Adimensional name " + std::to_string(i) + ": " + adimensionalNames[conductProtAdimensionals[i]] << std::endl;
        std::cout << "\t\t Range: " + adimensionalRanges[conductProtAdimensionals[i]].niceLabel() << std::endl;
        std::cout << "\t\t min: " + adimensionalRanges[conductProtAdimensionals[i]].getMin().niceLabel() << std::endl;
        std::cout << "\t\t max: " + adimensionalRanges[conductProtAdimensionals[i]].getMax().niceLabel() << std::endl;
    }


    // Select the conductance protocol using its ID (ID = 3) and all its features
    selectVoltageProtocol(protocolId);

    Measurement_t vHold;
    vHold.value = 2.0;
    vHold.prefix = UnitPfxMilli;
    vHold.unit = "V";
    setProtocolVoltage(0, vHold);

    Measurement_t vPulse;
    vPulse.value = 100.0;
    vPulse.prefix = UnitPfxMilli;
    vPulse.unit = "V";
    setProtocolVoltage(1, vPulse);

    Measurement_t vStep;
    vStep.value = 20.0;
    vStep.prefix = UnitPfxMilli;
    vStep.unit = "V";
    setProtocolVoltage(2, vStep);

    Measurement_t tHold;
    tHold.value = 5.0;
    tHold.prefix = UnitPfxMilli;
    tHold.unit = "s";
    setProtocolTime(0, tHold);

    Measurement_t tPulse;
    tPulse.value = 4.0;
    tPulse.prefix = UnitPfxMilli;
    tPulse.unit = "s";
    setProtocolTime(1, tPulse);

    Measurement_t adimN; // pulse number
    adimN.value = 10.0;
    adimN.prefix = UnitPfxNone;
    adimN.unit = "";
    setProtocolAdimensional(0, adimN);

    Measurement_t adimNR; // repetition number
    adimNR.value = 0.0;
    adimNR.prefix = UnitPfxNone;
    adimNR.unit = "";
    setProtocolAdimensional(1, adimNR);


    // Apply the chosen voltage protocol
    std::cout << "Start voltage protocol" << std::endl;
    ErrorCodes_t applyError = applyVoltageProtocol();
    if(applyError != Success){
        std::cout << "Cannot apply protocol"<< std::endl;
        disconnect();
        return -1;
    }

    // Define how much data save to file
    unsigned int myDataToRead = 12500; // 10s data at 1.25kHz
    unsigned int dataRead;
    uint16_t* buffer;
    QueueStatus_t status;
    unsigned int AccDataToRead = 0;

    purgeData();
    FILE* fid = fopen("myLog.dat", "wb");
    while(AccDataToRead <= myDataToRead){
        // Get information about the available data packets
        getQueueStatus(status);

        // Read packets and store to buffer
        if(status.availableDataPackets > 0){
            ErrorCodes_t readError = readData(status.availableDataPackets, dataRead, buffer);
            if(readError != Success){
                std::cout << "Error reading data" << std::endl;
                std::this_thread::sleep_for(std::chrono::milliseconds(10));
                continue;
            }
            AccDataToRead = AccDataToRead + dataRead;

            // Print data buffer to file
            fwrite(&buffer[0], 2, dataRead*(voltageChannelsNum+currentChannelsNum),fid);
        }

    }
    fclose(fid);

    std::cout << "Ended voltage protocol" << std::endl;
    std::cout << std::endl;

    // Test a feature not available in this device
    std::cout << "Test a feature not present in this device" << std::endl;
    uint16_t ledNums;
    ErrorCodes_t ledError = getLedsNumber(ledNums);
    if(ledError != Success){
        if(ledError == ErrorFeatureNotImplemented)
            std::cout << "Error feature not implemented: " << std::hex << ledError << std::endl;
        else{
            std::cout << "Another error: " << std::hex << ledError << std::endl;
        }
    } else {
        std::cout << "Led number: " << std::to_string(ledNums) << std::endl;
    }
    std::cout << std::endl;

    std::cout << "Disconnect the device" << std::endl;
    disconnect();

    return 0;

}
